// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef COMMON_H
#define COMMON_H

#include "degub.h"
#include "ini.h"
#include "container.h"
#include "stringCommons.h"
#include <string>
#include <sstream>
#include <iomanip>
#include <limits.h>

using namespace std;

//------------------------------MACROS & TYPEDEFS-------------------------------

#define ASMMAX 8
#define UNDEFINED_PPCWORD 0xBAADF33D
#define MAIN_MEMORY_SIZE 24*M
#define HARDWARE_MEMSIZE (0x8000)
#define CACHE_SIZE 256*K  //Level 2
#define CACHE_BASE 0x80000000
#define PHYSICALIZE(address) ((address) & ~0xC0000000)
#define MY_BACKBUFFERFORMAT D3DFMT_A8R8G8B8

#define CI(member) member(member) //class init
#define CCI(member) member(other.member)  //class copy init
#define CCO(member) member = other.member //class copy operator

#define VDEGUB if(g::verbose) DEGUB

#define MY_FWRITE(src, size, count, file) if(fwrite(src, size, count, file) != count) {\
	if(ferror(file)) { FAIL(UE_CANNOT_WRITE_FILE); }\
	else { FAIL(UE_EARLY_EOF); } }

//----------------------------------GENERIC-------------------------------------

//inline void strcpy_upper(char *dest, const char *src);
inline void memcpy_swaph(void *dest, const void *src, size_t len);
bool memeq_swapw(const void* a, const void* b, size_t len); //len must be a multiple of 4
inline void spbytes(char *buffer, __int64 bytes);

inline WORD swaph(WORD src);
inline DWORD swapw(DWORD src);
inline QWORD swapd(QWORD src);
inline float swapf(DWORD src);

inline string ips_string(QWORD cycles, DWORD elapsed_ms);

extern DWORD g_C87RC[4];  //Defined in opcode.cpp
inline DWORD getC87RC(DWORD fpscr);

inline bool is_power_of_2(DWORD i);
inline const char *abled(bool enabled);
inline const char *Abled(bool enabled);

//The last two parameters can be NULL and 0 if the bitmap data follows directly after the BMIH
void save_dib_to_png_file(const char *png_filename, LPBITMAPINFOHEADER dib,
													int dib_size, const void *pbits, int bitssize);
//A maximum of 9 digits. Uses TGLE.
bool getNextNumberedName(string& s, const char *beginning, int digits,
												 const char *ending);
bool dumpToNumberedFile(const void *data, size_t size, const char *beginning,
												int digits, const char *ending);
bool dumpToFile(const void *data, size_t size, const char *filename);
bool writePNG(const char *filename, WORD ByteCount, const void *data,
							int width, int height);
bool writePNGEx(const char *basename, int color_type, int width, int height,
								const void *data);
bool writePNGEx2(const string& basename, int color_type, int width, int height,
								 const void *data, int pitch, int transforms);

#define CONCAT(str1, str2) ((std::string(str1) + (str2)).c_str())

void PrintPacket(BYTE *buf, DWORD size);

inline void dump_spaces(int nspaces) {
	for(int i=0; i<nspaces; i++) {
		DEGUB(" ");
	}
}

union MYFILETIME {
	QWORD qword;
	FILETIME ft;
};

//--------------------------------BIG-ENDIAN-----------------------------------
//Big-endian style (Most significant bit is number 0)
inline BYTE makemaskb(int start, int end);
inline WORD makemaskh(int start, int end);
inline DWORD makemaskw(int start, int end);
inline QWORD makemaskd(int start, int end);
inline DWORD makeflag(DWORD bit);
inline WORD getbitsh(WORD hword, int start, int end);
inline DWORD getbitsw(DWORD dword, int start, int end);
inline QWORD getbitsd(QWORD qword, int start, int end);
inline DWORD exts(DWORD dword, int from);
inline bool getbit(DWORD dword, int bit);

inline void setbits(DWORD &dword, int start, int end, DWORD bits);

//------------------------------LITTLE-ENDIAN----------------------------------
inline bool getbitr(DWORD dword, int bit);
inline DWORD getbitsr(DWORD dword, int start, int end);
inline DWORD makeflagr(DWORD bit);

//------------------------------OPCODE HELPERS---------------------------------
inline bool signb(BYTE a);
inline bool signh(WORD a);
inline bool signw(DWORD a);
inline void setflags(DWORD &dword, DWORD flag, bool set);
inline bool getflag(DWORD dword, DWORD flag);
inline bool parityb(BYTE a); //even number of 1's
inline bool parityh(WORD a); //even number of 1's
inline bool parityw(DWORD a); //even number of 1's
inline bool carry(DWORD a, DWORD b);





#include "degub.h"
#include <stdlib.h>
#include <float.h>

//----------------------------------GENERIC-------------------------------------

/*inline void strcpy_upper(char *dest, const char *src) {
size_t i, len = strlen(src);

for(i=0; i<len; i++)
dest[i] = (char)toupper(src[i]);

dest[i] = 0;
}*/

inline DWORD make_24bits(const BYTE *_p) {
	return (_p[0] << 16) | (_p[1] << 8) | _p[2];
}

inline void memcpy_swaph(void *dest, const void *src, size_t len) {
	MYASSERT(len % 2 == 0);
	__asm {
		mov edi, dest;
		mov esi, src;
		mov ecx, len;
		shr ecx, 1;

loop1:
		lodsw;
		xchg al, ah;
		stosw;
		loop loop1;
	}
}

inline string spbytes(__int64 bytes) {
	const string bp[6] = {"","K","M","G","T","P"};
	__int64 temp = 1;
	int i;
	int decimals;

	for(i=0; i<5; i++) {
		if(bytes/temp < K)
			break;
		temp *= K;
	}

	decimals = 2;
	if(bytes/temp >= 10)
		decimals = 1;
	if(bytes/temp >= 100 || bytes < K)
		decimals = 0;

	ostringstream str;
	str << setiosflags(ios::fixed) << setprecision(decimals) <<
		float(bytes)/temp << bp[i] << "B";
	return str.str();
}


inline __declspec(naked) WORD __fastcall swaph(WORD) {
	__asm {
		mov eax, ecx;
		xchg al, ah;
		ret;
	}
	//alternate:
	//mov al, ch;
	//mov ah, cl;
}
__declspec(naked) inline DWORD __fastcall swapw(DWORD) {
	__asm {
		mov eax, ecx;
		bswap eax;
		ret;
	}
}
inline QWORD swapd(QWORD src) {  //Uncertain
	return swapw(DWORD(src >> 32)) | (QWORD(swapw((DWORD)src)) << 32);
}
inline float swapf(DWORD src) {
	DWORD d = swapw(src);
	return MAKE(float, d);
}

inline BYTE tswap(BYTE data) { return data; }
inline WORD tswap(WORD data) { return swaph(data); }
inline DWORD tswap(DWORD data) { return swapw(data); }
inline QWORD tswap(QWORD data) { return swapd(data); }

#define MEM_ACCESS_TYPES(macro) macro(b, BYTE) macro(h, WORD) macro(w, DWORD)\
	macro(d, QWORD)

inline string ips_string(QWORD cycles, DWORD elapsed_ms) {
	double ips = (double(cycles) / double(elapsed_ms)) * 1000;
	const int IPS_MAX = 1000;
	const char* prefix;
	if(ips < IPS_MAX) {
		prefix = "";
	} else {
		ips /= 1000;
		if(ips < IPS_MAX) {
			prefix = "K";
		} else {
			ips /= 1000;
			prefix = "M";
		}
	}
	ostringstream str;
	str << setiosflags(ios::fixed) << setprecision(4) << ips << prefix << "IPS";
	return str.str();
}

inline DWORD getC87RC(DWORD fpscr) {
	return g_C87RC[getbitsw(fpscr, 30, 31)];
}

inline void dumpFileTime(const char *name, QWORD qword) {
	ostringstream str;
	str << qword / 10000;
	DEGUB("%s: %s ms\n", name, make_space3(str.str()).c_str());
}

//Not naked because of breaky optimizations in the release version.
//Because this func is inline, it doesn't neccesarily expect the result in eax.
//Maybe there's something important in ebx, too.
//This should be stable.
inline bool is_power_of_2(DWORD i) {
	bool result;
	__asm {
		bsf eax, i;
		jz not_power;
		bsr ebx, i;
		cmp eax, ebx;
		jne not_power; //can be optimized using setcc
		//is_power:
		mov result, 1;
		jmp end;
not_power:
		mov result, 0;
end:
	}
	return result;
}

inline const char *abled(bool enabled) {
	return enabled ? "enabled" : "disabled";
}
inline const char *Abled(bool enabled) {
	return enabled ? "Enabled" : "Disabled";
}
inline const char* s(UINT number) {
	return number == 1 ? "" : "s";
}

//sign-extends a 6-bit signed integer to 8 bits
inline char exts6(char i) {
	__asm {
		test i, 0x20;
		jz end;
		//signed:
		or i, 0xC0;
end:
	}
	return i;
}

inline void memset_dword(void *dest, DWORD data, size_t ndwords) {
	__asm {
		mov edi, dest;
		mov eax, data;
		mov ecx, ndwords;
		rep stosd;
	}
}

inline bool MyResumeThread(HANDLE hThread) {
	int res;
	do {
		res = ResumeThread(hThread);
	} while(res > 0);
	return true;
}

//---------------------------------BIG-ENDIAN-----------------------------------

inline BYTE makemaskb(int start, int end) {
	return (BYTE)_rotl((2 << (end - start)) - 1, 7 - end);
}
inline WORD makemaskh(int start, int end) {
	return (WORD)_rotl((2 << (end - start)) - 1, 15 - end);
}
inline DWORD makemaskw(int start, int end) {
	return _rotl((2 << (end - start)) - 1, 31 - end);
}
inline QWORD makemaskd(int start, int end) {
	return _rotl64((QWORD(2) << (end - start)) - 1, 63 - end);
}
inline DWORD makeflag(DWORD bit) {
	return 0x80000000 >> bit;
}
inline BYTE getbitsb(BYTE byte, int start, int end) {
	return (byte & makemaskb(start, end)) >> BYTE(7 - end);
}
inline WORD getbitsh(WORD hword, int start, int end) {
	return (hword & makemaskh(start, end)) >> WORD(15 - end);
}
inline DWORD getbitsw(DWORD dword, int start, int end) {
	return (dword & makemaskw(start, end)) >> (31 - end);
}
inline QWORD getbitsd(QWORD qword, int start, int end) {
	return (qword & makemaskd(start, end)) >> (63 - end);
}
inline DWORD exts(DWORD dword, int from) {
	if(getbit(dword, from))
		return dword | makemaskw(0, from);
	else
		return dword;
}
inline bool getbit(DWORD dword, int bit) {
	return (dword & makeflag(bit)) != 0;
}
inline void setbits(DWORD &dword, int start, int end, DWORD bits) {
	dword = (dword & ~makemaskw(start, end)) | 
		((bits << (31 - end)) & makemaskw(start, end));
}

//-------------------------------LITTLE-ENDIAN----------------------------------

inline bool getbitr(DWORD dword, int bit) {
	return (dword & makeflagr(bit)) != 0;
}
inline DWORD getbitsr(DWORD dword, int start, int end) {
	return (dword & makemaskw(31-start, 31-end)) >> (end);
}
inline DWORD makeflagr(DWORD bit) {
	return 1 << bit;
}
/*inline DWORD extsr(DWORD dword, int from) {
if(getbitr(dword, from))
return dword | makemaskw(0, 31 - from);
else
return dword;
}*/

//---------------------------------ANY ENDIAN-----------------------------------

inline bool signb(BYTE a) { return (a & 0x80) != 0; }
inline bool signh(WORD a) { return (a & 0x8000) != 0; }
inline bool signw(DWORD a) { return (a & 0x80000000) != 0; }

//Sets or clears a flag in the specified DWORD
inline void setflags(DWORD &dword, DWORD flags, bool set) {
	if(set)
		dword |= flags;
	else
		dword &= ~flags;
}

inline bool getflag(DWORD dword, DWORD flag) {
	return (dword & flag) != 0;
}

inline bool parityb(BYTE a) {
	int p=0;
	for(int i=0; i<8; i++) {
		if((a >> i) & 0x01)
			p++;
	}
	return !(p & 1);
}

inline bool parityh(WORD a) {
	int p=0;
	for(int i=0; i<16; i++) {
		if((a >> i) & 0x01)
			p++;
	}
	return !(p & 1);
}

inline bool parityw(DWORD a) {
	int p=0;
	for(int i=0; i<32; i++) {
		if((a >> i) & 0x01)
			p++;
	}
	return !(p & 1);
}

__declspec(naked) inline bool __fastcall carry(DWORD, DWORD) {
	__asm {
		add ecx, edx;
		setc al; //assumes sizeof(bool) == 1
		ret;
	}
}

#endif	//COMMON_H
